<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash; // for password hashing
use App\Models\User; // import User model

class AuthController extends Controller
{
    // Show Register form
    public function showUserRegisterForm()
    {
        return view('auth.user-register');
    }

    // Handle User Registration
    public function userRegister(Request $request)
    {
        $request->validate([
            'name'=>'required|string|max:255',
            'email'=>'required|string|email|max:255|unique:users',
            'password'=>'required|string|min:8|confirmed', // must have password_confirmation field
        ]);

        User::create([
            'name' => $request->name,
            'email' => $request->email,
            'password' => Hash::make($request->password),
        ]);

        return redirect()->route('login')->with('success','Registration Successful. Please Login.');
    }

    // Show Login form
    public function showUserLoginForm()
    {
        return view('auth.user-login');
    }

    // Handle User Login
    public function userLogin(Request $request)
    {
        $credentials = $request->validate([
            'email'=>'required|string|email',
            'password'=>'required|string',
        ]);

        if (Auth::attempt($credentials)) {
            $request->session()->regenerate();
            return redirect()->intended(route('user.dashboard'));
        }

        return back()->withErrors([
            'email'=>'The provided credentials do not match our records',
        ]);
    }

    // User Dashboard
    public function userDashboard()
    {
        return view('auth.dashboard');
    }

    // Logout
    public function logout(Request $request)
    {
        Auth::logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect()->route('login')->with('success','Logged out successfully.');
    }
}
